<?php
/* --------------------------------------------------------------
 AdminServiceProviderRegistration.php 2021-07-09
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2021 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

declare(strict_types=1);

namespace Gambio\Admin\Application\Bootstrapper;

use Gambio\Admin\Application\ErrorHandling\AdminErrorServiceProvider;
use Gambio\Admin\Application\Http\AdminHttpServiceProvider;
use Gambio\Admin\Layout\Favorites\FavoritesServiceProvider;
use Gambio\Admin\Layout\Menu\AdminMenuServiceProvider;
use Gambio\Admin\Layout\Renderer\AdminRendererServiceProvider;
use Gambio\Core\Application\Application;
use Gambio\Core\Application\Bootstrapper;
use Gambio\Core\Application\ServiceProviderRegistry\ServiceProviderRegistry;
use Gambio\Core\Application\ServiceProviderRegistry\ServiceProviderRegistryCache;
use Gambio\Core\Application\ServiceProviderRegistry\ServiceProviderRegistryType;
use Gambio\Core\Application\ValueObjects\Environment;
use Gambio\Core\Cache\Services\CacheFactory;
use RuntimeException;

/**
 * Class AdminServiceProviderRegistration
 *
 * @package Gambio\Admin\Application\Bootstrapper
 */
class AdminServiceProviderRegistration implements Bootstrapper
{
    private const CORE_SERVICE_PROVIDERS = [
        FavoritesServiceProvider::class,
        AdminRendererServiceProvider::class,
        AdminMenuServiceProvider::class,
        AdminHttpServiceProvider::class,
        AdminErrorServiceProvider::class,
    ];
    
    
    /**
     * @inheritDoc
     */
    public function boot(Application $application): void
    {
        $this->registerCoreServiceProvider($application);
        
        $cacheFactory = $this->getCacheFactory($application);
        $environment  = $this->getEnvironment($application);
        
        $registryCache = new ServiceProviderRegistryCache($cacheFactory);
        $registry      = new ServiceProviderRegistry($registryCache, $environment);
        $registryType  = ServiceProviderRegistryType::admin();
        
        foreach ($registry->getRegistry($registryType) as $adminModuleProvider) {
            $application->registerProvider($adminModuleProvider);
        }
    }
    
    
    /**
     * Registers all of the core admin service providers.
     *
     * @param Application $application
     */
    private function registerCoreServiceProvider(Application $application): void
    {
        $this->registerServiceProviders(self::CORE_SERVICE_PROVIDERS, $application);
    }
    
    
    /**
     * Tries to register service providers by using the provided list.
     * This method silently continues if anything fails on the registration process.
     *
     * @param array       $serviceProviders
     * @param Application $application
     */
    private function registerServiceProviders(array $serviceProviders, Application $application): void
    {
        foreach ($serviceProviders as $provider) {
            $application->registerProvider($provider);
        }
    }
    
    
    /**
     * @param Application $application
     *
     * @return CacheFactory
     */
    private function getCacheFactory(Application $application): CacheFactory
    {
        if (!$application->has(CacheFactory::class)) {
            throw new RuntimeException(CacheFactory::class . ' must be registered!');
        }
        
        return $application->get(CacheFactory::class);
    }
    
    
    /**
     * @param Application $application
     *
     * @return Environment
     */
    private function getEnvironment(Application $application): Environment
    {
        if (!$application->has(Environment::class)) {
            throw new RuntimeException(Environment::class . ' must be registered!');
        }
        
        return $application->get(Environment::class);
    }
}